/*
 * Copyright (C) 2012-2025 Japan Smartphone Security Association
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.jssec.android.sharedmemory.inhouseservice.messenger;

import android.app.Activity;
import android.content.ComponentName;
import android.content.Context;
import android.content.Intent;
import android.content.ServiceConnection;
import android.os.Bundle;
import android.os.Handler;
import android.os.IBinder;
import android.os.Message;
import android.os.Messenger;
import android.os.RemoteException;
import android.os.SharedMemory;
import android.system.ErrnoException;
import android.widget.Toast;
import android.util.Log;

import org.jssec.android.shared.PkgCert;
import org.jssec.android.shared.SigPerm;
import org.jssec.android.shared.Utils;

import java.nio.ByteBuffer;
import java.nio.ReadOnlyBufferException;

import static android.system.OsConstants.PROT_EXEC;
import static android.system.OsConstants.PROT_READ;
import static android.system.OsConstants.PROT_WRITE;

public class MainActivity extends Activity {

    private final String TAG = "SHMClient";

    // Messenger used for sending data to Service
    private Messenger mServiceMessenger = null;

    // SharedMemory objects
    private SharedMemory myShared1;
    private SharedMemory myShared2;
    // ByteBuffers for mapping SharedMemories
    private ByteBuffer mBuf1;
    private ByteBuffer mBuf2;

    // Information of using Activity
    private static final String SHM_PACKAGE =
        "org.jssec.android.sharedmemory.inhouseservice.messenger";
    private static final String SHM_CLASS =
        "org.jssec.android.sharedmemory.inhouseservice.messenger.SHMService";

    // In-house Signature Permission
    private static final String MY_PERMISSION =
        "org.jssec.android.sharedmemory.inhouseservice.messenger.MY_PERMISSION";

    // Hash value of the certification of In-house applications
    private static String sMyCertHash = null;
    private static String myCertHash(Context context) {
        if (sMyCertHash == null) {
            if (Utils.isDebuggable(context)) {
                // Hash value of the certificate "androiddebugkey" stored in
                // debug.keystore
                sMyCertHash = "0EFB7236 328348A9 89718BAD DF57F544 D5CCB4AE B9DB34BC 1E29DD26 F77C8255";
            } else {
                // Hash value of the certificate "my company key" in keystore
                sMyCertHash = "D397D343 A5CBC10F 4EDDEB7C A10062DE 5690984F 1FB9E88B D7B3A7C2 42E142CA";
            }
        }
        return sMyCertHash;
    }

    // true iff connecting to Service
    private boolean mIsBound = false;

    // Handler handling Messages received from Server
    private class MyHandler extends Handler {
        @Override
        public void handleMessage(Message msg) {
            switch (msg.what) {
            case SHMService.SHMEM1:
                // SHMEM 1 is provided from Service
                // ShareMemory object is stored in Message.obj
                Log.d(TAG, "got SHMEM1");
                myShared1 = (SharedMemory) msg.obj;
                useSHMEM1();
                break;
            case SHMService.SHMEM2:
                // SHMEM2 is provided from Service
                Log.d(TAG, "got SHMEM2");
                myShared2 = (SharedMemory) msg.obj;
                useSHMEM2();
                break;
            case SHMService.MSG_END:
                Log.d(TAG, "got MSG_END");
                alloverNow();
                break;
            default:
                Log.e(TAG, "invalid message: " + msg.what);
            }
        }
    }

    private Handler mHandler = new MyHandler();

    // Messanger used when receiving data from Service
    private Messenger mLocalMessenger = new Messenger(mHandler);

    // Connection used for connecting to Service
    // This is needed if implementation uses bindService
    private class MyServiceConnection implements ServiceConnection {
        // called when connected with Service
        public void onServiceConnected(ComponentName className, IBinder service){
            mServiceMessenger = new Messenger(service);
            // When bound to SharedMemory Service, request 1st SharedMemory
            sendMessageToService(SHMService.MSG_ATTACH);
        }
        // This is called when Service unexpectedly terminate and connection is
        // broken
        public void onServiceDisconnected(ComponentName className){
            mIsBound = false;
            mServiceMessenger = null;
        }
    }
    private MyServiceConnection mServiceConnection;

    @Override
    protected void onCreate(Bundle savedInstanceState) {
        super.onCreate(savedInstanceState);
        setContentView(R.layout.activity_main);
        doBindService ();
    }

    // Connect to Shared Memory Service
    private void doBindService () {
        mServiceConnection = new MyServiceConnection();
        if (!mIsBound) {
            // *** POINT 9 *** Verify that the in-house-defined signature
            // permission is defined by the in-house application.
            if (!SigPerm.test(this, MY_PERMISSION, myCertHash(this))) {
                Toast.makeText(this,  "In-house signature permission is not defined by an in-house application.", Toast.LENGTH_LONG).show();
                return;
            }
            // *** POINT 10 *** Verify that the destination application is signed
            // by the in-house certificate.
            if (!PkgCert.test(this, SHM_PACKAGE, myCertHash(this))) {
                Toast.makeText(this, "Binding Service is not an in-house application.", Toast.LENGTH_LONG).show();
                return;
            }
        }
        Intent it = new Intent();
        // *** POINT 11 *** Sensitive information can be sent because the
        // destination application is in-house.
        it.putExtra("PARAM", "Sensitive Information");

        // *** POINT 12 *** Use explicit Intent to call an in-house service
        it.setClassName(SHM_PACKAGE, SHM_CLASS);

        if (!bindService(it, mServiceConnection, Context.BIND_AUTO_CREATE)) {
            Toast.makeText(this, "Bind Service Failed", Toast.LENGTH_LONG).show();
            return;
        }
        mIsBound = true;
    }

    // Unbind connection with Service
    private void releaseService () {
        unbindService(mServiceConnection);
    }

    // An example of using SHMEM1
    private void useSHMEM1 () {
        // Because only read access is permitted for SHMEM1, mapping with
        // different protection mode will raise an exception.
        // The exception will be handled by mapMemory()
        mBuf1 = mapMemory(myShared1, PROT_WRITE, SHMService.SHMEM1_BUF1_OFFSET,
                          SHMService.SHMEM1_BUF1_LENGTH);
        // map with PROT_READ
        mBuf1 = mapMemory(myShared1, PROT_READ, SHMService.SHMEM1_BUF1_OFFSET,
                          SHMService.SHMEM1_BUF1_LENGTH);
        // Read data which Service side set
        int len = mBuf1.getInt();
        byte[] bytes = new byte[len];
        mBuf1.get(bytes);
        String message = new String(bytes);
        Toast.makeText(MainActivity.this,
                       "Got: " + message, Toast.LENGTH_LONG).show();
        // Because the buffer is read only, writing will cause
        // ReadOnlyBufferException
        try {
            mBuf1.putInt(0);
        } catch (ReadOnlyBufferException e){
            Log.e(TAG, "Write to read only buffer: " + e.getMessage());
        }
        // Reply to Service
        sendMessageToService(SHMService.MSG_REPLY1);
        // then, request a SharedMemory with write permission
        sendMessageToService(SHMService.MSG_ATTACH2);
    }

    // An example of using SHMEM2
    private void useSHMEM2 () {
        // We are allowed to write into SHMEM2, map it with PROT_WRITE
        // Service side set SHMEM2 as PROT_WRITE, so mapping with
        // PROT_READ | PROT_WRITE will raise an exception
        mBuf2 = mapMemory(myShared2, PROT_WRITE, SHMService.SHMEM2_BUF1_OFFSET,
                          SHMService.SHMEM2_BUF1_LENGTH);
        if (mBuf2 != null) {
            // Even if the protection mode is PROT_WRITE only, it will also be
            // readable on most SoC.
            int size = mBuf2.getInt();
            byte [] bytes = new byte[size];
            mBuf2.get(bytes);
            String msg = new String(bytes);
            Log.d(TAG, "Got a message from service: " + msg);

            // Accessing outside of the mapped region will cause
            // IndexOutOfBoundsException
            try {
                mBuf2.get(SHMService.SHMEM2_BUF1_LENGTH + 1);
            } catch (IndexOutOfBoundsException e){
                Log.e(TAG, "out of bound: " + e.getMessage());
            }

            // Override the data which Service side set before
            String replyStr = "OK Thanks!";
            mBuf2.putInt(replyStr.length());
            mBuf2.put(replyStr.getBytes());
            // Reply to Service
            sendMessageToService(SHMService.MSG_REPLY2);
        }
    }

    // Map specified SharedMemory
    private ByteBuffer mapMemory(SharedMemory mem, int proto, int offset,
                                 int length){
        ByteBuffer tempBuf;
        try {
            tempBuf = mem.map(proto, offset, length);
        } catch (ErrnoException e){
            Log.e(TAG,"could not map, proto: " + proto + ", offset:" +
                       offset +", length: " + length + "\n " + e.getMessage() +
                       "err no. = " + e.errno);
            return null;
        }
        return tempBuf;
    }

    // Reply message to Server
    private void sendMessageToService(int what){
        try {
            Message msg = Message.obtain();
            msg.what = what;
            msg.replyTo = mLocalMessenger;
            mServiceMessenger.send(msg);
        } catch (RemoteException e) {
            Log.e(TAG, "Error in sending message: " + e.getMessage());
        }
    }

    // Finalize no more used SharedMemory
    private void alloverNow() {
        // Notify Service that we are done
        sendMessageToService(SHMService.MSG_DETACH);
        sendMessageToService(SHMService.MSG_DETACH2);
        // unmap ByteBuffers
        if (mBuf1 != null) SharedMemory.unmap(mBuf1);
        if (mBuf2 != null) SharedMemory.unmap(mBuf2);
        // Close SharedMemory
        myShared1.close();
        myShared2.close();
        mBuf1 = null;
        mBuf2 = null;
        myShared1 = null;
        myShared2 = null;
        // Disconnect from Service
        releaseService();
    }
}

